/*
 * Decompiled with CFR 0.152.
 */
package org.apache.geode.internal;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.geode.annotations.VisibleForTesting;
import org.apache.geode.annotations.internal.MakeNotStatic;
import org.apache.geode.internal.ClassPathLoader;
import org.apache.geode.internal.DeployedJar;
import org.apache.geode.logging.internal.log4j.api.LogService;
import org.apache.logging.log4j.Logger;

public class JarDeployer
implements Serializable {
    private static final long serialVersionUID = 1L;
    private static final Logger logger = LogService.getLogger();
    private static final Pattern POUND_VERSION_SCHEME = Pattern.compile("^vf\\.gf#(?<artifact>.*)\\.jar#(?<version>\\d+)$");
    static final Pattern DEPLOYED_FILE_PATTERN = Pattern.compile("(?<baseName>..*)\\.v(?<version>\\d++).jar$");
    private static final Pattern USER_VERSION_PATTERN = Pattern.compile("(?<artifact>.*?)[-.]\\d+.*\\.jar$");
    @MakeNotStatic
    private static final Lock lock = new ReentrantLock();
    private final Map<String, DeployedJar> deployedJars = new ConcurrentHashMap<String, DeployedJar>();
    private final File deployDirectory;

    public JarDeployer() {
        this(new File(System.getProperty("user.dir")));
    }

    public JarDeployer(File deployDirectory) {
        this.deployDirectory = deployDirectory;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DeployedJar deployWithoutRegistering(File stagedJar) throws IOException {
        lock.lock();
        String stagedJarName = stagedJar.getName();
        String artifactId = JarDeployer.getArtifactId(stagedJarName);
        try {
            boolean shouldDeployNewVersion = this.shouldDeployNewVersion(artifactId, stagedJar);
            if (!shouldDeployNewVersion) {
                logger.debug("No need to deploy a new version of {}", (Object)stagedJarName);
                DeployedJar deployedJar = null;
                return deployedJar;
            }
            this.verifyWritableDeployDirectory();
            Path deployedFile = this.getNextVersionedJarFile(stagedJarName).toPath();
            Files.copy(stagedJar.toPath(), deployedFile, new CopyOption[0]);
            DeployedJar deployedJar = new DeployedJar(deployedFile.toFile());
            return deployedJar;
        }
        finally {
            lock.unlock();
        }
    }

    public List<DeployedJar> findDeployedJars() {
        return this.getDeployedJars().values().stream().collect(Collectors.toList());
    }

    public void suspendAll() {
        lock.lock();
    }

    public void resumeAll() {
        lock.unlock();
    }

    protected File getNextVersionedJarFile(String unversionedJarName) {
        int maxVersion = this.getMaxVersion(JarDeployer.getArtifactId(unversionedJarName));
        String nextVersionJarName = FilenameUtils.getBaseName(unversionedJarName) + ".v" + (maxVersion + 1) + ".jar";
        logger.debug("Next versioned jar name for {} is {}", (Object)unversionedJarName, (Object)nextVersionJarName);
        return new File(this.deployDirectory, nextVersionJarName);
    }

    protected int getMaxVersion(String artifactId) {
        return Arrays.stream(this.deployDirectory.list()).filter(x -> artifactId.equals(JarDeployer.toArtifactId(x))).map(JarDeployer::extractVersionFromFilename).reduce(Integer::max).orElse(0);
    }

    public static int extractVersionFromFilename(String filename) {
        Matcher matcher = DEPLOYED_FILE_PATTERN.matcher(filename);
        if (matcher.find()) {
            return Integer.parseInt(matcher.group(2));
        }
        return 0;
    }

    public static boolean isDeployedFile(String filename) {
        return DEPLOYED_FILE_PATTERN.matcher(filename).find();
    }

    public static boolean isSemanticVersion(String filename) {
        return USER_VERSION_PATTERN.matcher(filename).find();
    }

    static String toArtifactId(String sequencedJarFileName) {
        String baseName = JarDeployer.getDeployedFileBaseName(sequencedJarFileName);
        if (baseName == null) {
            return null;
        }
        return JarDeployer.getArtifactId(baseName + ".jar");
    }

    public static String getArtifactId(String deployedJarFileName) {
        Matcher semanticVersionMatcher = USER_VERSION_PATTERN.matcher(deployedJarFileName);
        if (semanticVersionMatcher.matches()) {
            return semanticVersionMatcher.group("artifact");
        }
        return FilenameUtils.getBaseName(deployedJarFileName);
    }

    public static String getDeployedFileBaseName(String sequencedJarFileName) {
        Matcher semanticVersionMatcher = DEPLOYED_FILE_PATTERN.matcher(sequencedJarFileName);
        if (semanticVersionMatcher.matches()) {
            return semanticVersionMatcher.group("baseName");
        }
        return null;
    }

    public void verifyWritableDeployDirectory() throws IOException {
        try {
            if (this.deployDirectory.canWrite()) {
                return;
            }
        }
        catch (SecurityException ex) {
            throw new IOException("Unable to write to deploy directory", ex);
        }
        throw new IOException("Unable to write to deploy directory: " + this.deployDirectory.getCanonicalPath());
    }

    protected void renameJarsWithOldNamingConvention() throws IOException {
        Set<File> jarsWithOldNamingConvention = this.findJarsWithOldNamingConvention();
        if (jarsWithOldNamingConvention.isEmpty()) {
            return;
        }
        for (File jar : jarsWithOldNamingConvention) {
            this.renameJarWithOldNamingConvention(jar);
        }
    }

    protected Set<File> findJarsWithOldNamingConvention() {
        return Stream.of(this.deployDirectory.listFiles()).filter(file -> this.isOldNamingConvention(file.getName())).collect(Collectors.toSet());
    }

    protected boolean isOldNamingConvention(String fileName) {
        return POUND_VERSION_SCHEME.matcher(fileName).matches();
    }

    private void renameJarWithOldNamingConvention(File oldJar) throws IOException {
        Matcher matcher = POUND_VERSION_SCHEME.matcher(oldJar.getName());
        if (!matcher.matches()) {
            throw new IllegalArgumentException("The given jar " + oldJar.getCanonicalPath() + " does not match the old naming convention");
        }
        String unversionedJarNameWithoutExtension = matcher.group(1);
        String jarVersion = matcher.group(2);
        String newJarName = unversionedJarNameWithoutExtension + ".v" + jarVersion + ".jar";
        File newJar = new File(this.deployDirectory, newJarName);
        logger.debug("Renaming deployed jar from {} to {}", (Object)oldJar.getCanonicalPath(), (Object)newJar.getCanonicalPath());
        FileUtils.moveFile(oldJar, newJar);
    }

    public void loadPreviouslyDeployedJarsFromDisk() {
        logger.info("Loading previously deployed jars");
        lock.lock();
        try {
            this.verifyWritableDeployDirectory();
            this.renameJarsWithOldNamingConvention();
            Map<String, Integer> artifactToMaxVersion = this.findArtifactsAndMaxVersion();
            ArrayList<DeployedJar> latestVersionOfEachJar = new ArrayList<DeployedJar>();
            for (File file : this.deployDirectory.listFiles()) {
                String artifactId = JarDeployer.toArtifactId(file.getName());
                if (artifactId == null) continue;
                int version = JarDeployer.extractVersionFromFilename(file.getName());
                if (version < artifactToMaxVersion.get(artifactId)) {
                    FileUtils.deleteQuietly(file);
                    continue;
                }
                latestVersionOfEachJar.add(new DeployedJar(file));
            }
            this.registerNewVersions(latestVersionOfEachJar);
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        finally {
            lock.unlock();
        }
    }

    Map<String, Integer> findArtifactsAndMaxVersion() {
        HashMap<String, Integer> artifactToMaxVersion = new HashMap<String, Integer>();
        for (String fileName : this.deployDirectory.list()) {
            String artifactId = JarDeployer.toArtifactId(fileName);
            if (artifactId == null) continue;
            int version = JarDeployer.extractVersionFromFilename(fileName);
            Integer maxVersion = (Integer)artifactToMaxVersion.get(artifactId);
            if (maxVersion != null && maxVersion >= version) continue;
            artifactToMaxVersion.put(artifactId, version);
        }
        return artifactToMaxVersion;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<DeployedJar> registerNewVersions(List<DeployedJar> deployedJars) throws ClassNotFoundException {
        lock.lock();
        try {
            HashMap<DeployedJar, DeployedJar> newVersionToOldVersion = new HashMap<DeployedJar, DeployedJar>();
            for (DeployedJar deployedJar : deployedJars) {
                if (deployedJar == null) continue;
                logger.info("Registering new version of jar: {}", (Object)deployedJar);
                DeployedJar oldJar = this.deployedJars.put(deployedJar.getArtifactId(), deployedJar);
                ClassPathLoader.getLatest().chainClassloader(deployedJar);
                newVersionToOldVersion.put(deployedJar, oldJar);
            }
            for (Map.Entry entry : newVersionToOldVersion.entrySet()) {
                DeployedJar newjar = (DeployedJar)entry.getKey();
                DeployedJar oldJar = (DeployedJar)entry.getValue();
                newjar.registerFunctions();
                if (oldJar == null) continue;
                oldJar.cleanUp(newjar);
            }
        }
        finally {
            lock.unlock();
        }
        return deployedJars;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<DeployedJar> deploy(Set<File> stagedJarFiles) throws IOException, ClassNotFoundException {
        ArrayList<DeployedJar> deployedJars = new ArrayList<DeployedJar>(stagedJarFiles.size());
        for (File jar : stagedJarFiles) {
            if (DeployedJar.hasValidJarContent(jar)) continue;
            throw new IllegalArgumentException("File does not contain valid JAR content: " + jar.getName());
        }
        lock.lock();
        try {
            for (File stagedJarFile : stagedJarFiles) {
                deployedJars.add(this.deployWithoutRegistering(stagedJarFile));
            }
            List<DeployedJar> list = this.registerNewVersions(deployedJars);
            return list;
        }
        finally {
            lock.unlock();
        }
    }

    private boolean shouldDeployNewVersion(String artifactId, File stagedJar) throws IOException {
        DeployedJar oldDeployedJar = this.deployedJars.get(artifactId);
        if (oldDeployedJar == null) {
            return true;
        }
        if (oldDeployedJar.hasSameContentAs(stagedJar)) {
            logger.warn("Jar is identical to the latest deployed version: {}", (Object)oldDeployedJar.getFileCanonicalPath());
            return false;
        }
        return true;
    }

    @VisibleForTesting
    public DeployedJar getDeployedJar(String jarName) {
        return this.deployedJars.get(JarDeployer.getArtifactId(jarName));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    public DeployedJar deploy(File stagedJarFile) throws IOException, ClassNotFoundException {
        lock.lock();
        HashSet<File> jarFiles = new HashSet<File>();
        jarFiles.add(stagedJarFile);
        try {
            List<DeployedJar> deployedJars = this.deploy(jarFiles);
            if (deployedJars == null || deployedJars.size() == 0) {
                DeployedJar deployedJar = null;
                return deployedJar;
            }
            DeployedJar deployedJar = deployedJars.get(0);
            return deployedJar;
        }
        finally {
            lock.unlock();
        }
    }

    public Map<String, DeployedJar> getDeployedJars() {
        return Collections.unmodifiableMap(this.deployedJars);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String undeploy(String jarName) throws IOException {
        String artifactId = JarDeployer.getArtifactId(jarName);
        lock.lock();
        try {
            DeployedJar deployedJar = this.deployedJars.get(artifactId);
            if (deployedJar == null) {
                throw new IllegalArgumentException(jarName + " not deployed");
            }
            if (!deployedJar.getDeployedFileName().equals(jarName)) {
                throw new IllegalArgumentException(jarName + " not deployed");
            }
            this.deployedJars.remove(artifactId);
            ClassPathLoader.getLatest().unloadClassloaderForArtifact(artifactId);
            deployedJar.cleanUp(null);
            this.deleteAllVersionsOfJar(jarName);
            String string = deployedJar.getFileCanonicalPath();
            return string;
        }
        finally {
            lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteAllVersionsOfJar(String jarName) {
        lock.lock();
        String artifactId = JarDeployer.getArtifactId(jarName);
        try {
            for (File file : this.deployDirectory.listFiles()) {
                if (!artifactId.equals(JarDeployer.toArtifactId(file.getName()))) continue;
                logger.info("Deleting: {}", (Object)file.getAbsolutePath());
                FileUtils.deleteQuietly(file);
            }
        }
        finally {
            lock.unlock();
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder(this.getClass().getName());
        sb.append('@').append(System.identityHashCode(this)).append('{');
        sb.append("deployDirectory=").append(this.deployDirectory);
        sb.append('}');
        return sb.toString();
    }
}

