/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.commons.util;


import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;

import java.util.Date;
import java.util.Locale;

import javax.faces.application.Application;
import javax.faces.context.FacesContext;
import javax.faces.el.ValueBinding;



/**
 * Utility class for Tag classes
 *
 * @version $Name:  $ ($Revision: adfrt/faces/adf-faces-impl/src/main/java/oracle/adfinternal/view/faces/taglib/util/TagUtils.java#1 $) $Date: 11-nov-2005.14:59:38 $
 *
 */
public final class TagUtils
{
  private static final Log LOG = LogFactory.getLog(TagUtils.class);

  private TagUtils()
  {
  }

  public static ValueBinding getValueBinding(String valueBindingExpression)
  {
    FacesContext context = FacesContext.getCurrentInstance();
    Application app = context.getApplication();
    return app.createValueBinding(valueBindingExpression);
  }

  public static void assertNotNull(Object object)
  {
    if (null == object)
      throw new NullPointerException();
  }

  // Helpful with tag auto generation. Though this isn't really required.
  /**
   * Return the same string. It is there for convenience and makes life easy
   * while auto generating tags.
   * @param value
   * @return
   */
  public static String getString(
    String      value)
  {
    return value;
  }

  /**
   * String --> boolean
   * @param value
   * @return
   */
  public static boolean getBoolean(
    String      value)
  {
    return Boolean.valueOf(value).booleanValue();
  }

  /**
   * String --> int
   * @param value
   * @return
   */
  public static int getInteger(
    String      value)
  {
    return Integer.valueOf(value).intValue();

  }

  /**
   * String --> long
   * @param value
   * @return
   */
  public static long getLong(
    String      value)
  {
    return Long.valueOf(value).longValue();

  }

  /**
   * String --> long
   * @param value
   * @return
   */
  public static double getDouble(
    String      value)
  {
    return Double.valueOf(value).doubleValue();

  }

  /**
   * String --> long
   * @param value
   * @return
   */
  public static float getFloat(
    String      value)
  {
    return Float.valueOf(value).floatValue();

  }

  /**
   * These are normally NMTOKEN type in attributes
   * String --> String[]
   * @param value
   * @return

  public static String[] getStringArray(
    String      value) throws ParseException
  {
    return getTokensArray(value);
  }
  */

  /**
   *  ISO Date String --> Date
   * @param value
   * @return
   */
  public static Date getDate(
    String      value)
  {
     return parseISODate(value);
  }

  /**
   * String --> Locale
   * @param value
   * @return
   */
  public static Locale getLocale(
    String      value)
  {
    return getLocaleInternal(value);
  }

  /**
   * String --> TimeZone
   * @param value
   * @return

  public static TimeZone getTimeZone(
    String value)
  {
    return DateUtils.getSupportedTimeZone(value);
  }
   */

  public static boolean isValueReference(String expression)
  {
    if (null != expression)
    {
      int start = expression.indexOf("#{");
      if ((start >= 0) && (expression.indexOf('}', start + 1) >= 0))
        return true;
    }

    return false;
  }



  /**
   * Takes a string that is a composite of tokens, extracts tokens delimited
   *  by any whitespace character sequence combination and returns a String
   *  array of such tokens.
   * @throws ParseException In case of invalid character in the specified
   *           composite. The only invalid character is a comma (',').

  private static String[] _getTokensArray(String tokenComposite)
    throws ParseException
  {
    if (tokenComposite == null || "".equals(tokenComposite))
      return null;

    return XMLUtils.parseNameTokens(tokenComposite);
  }
  */

  /**
   * Parse a string into a java.util.Date object.  The
   * string must be in ISO 9601 format (yyyy-MM-dd).
   * @todo why not throw the exception in a different format?
   *       why do we kill it here and return null?
   */
  static private final Date parseISODate(String stringValue)
  {
    try
    {
      return getDateFormat().parse(stringValue);
    }
    catch (ParseException pe)
    {
      if (LOG.isInfoEnabled()) {
        LOG.info("CANNOT_PARSE_VALUE_INTO_DATE_WITH_YYYY_MM_DD_PATTERN "+ stringValue, pe);
      }
      return null;
    }
  }

  private static Locale getLocaleInternal(String locale)
  {
    String localeStr = locale.replace('-','_');
    String[] tokens = localeStr.split("[_]", 3);
    Locale locl = null;

    if ( tokens.length == 1)
    {
      locl = new Locale(tokens[0]); //lang
    }
    else if (tokens.length == 2)
    {
      locl = new Locale(tokens[0], tokens[1]); // lang + country
    }
    else if (tokens.length == 3 )
    {
      locl = new Locale(tokens[0], tokens[1], tokens[2]); // lang + country + variant
    }
    else
    {
      if(LOG.isWarnEnabled())
        LOG.warn("tokens length should not be greater than 3.");
    }
    return locl;
  }

  // We rely strictly on ISO 8601 formats
  private static DateFormat getDateFormat()
  {
    return new SimpleDateFormat("yyyy-MM-dd");
  }
}
