/*
 * Decompiled with CFR 0.152.
 */
package org.apache.airavata.sharing.registry.db.utils;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Stack;
import java.util.concurrent.Semaphore;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionPool {
    private static final Logger logger = LoggerFactory.getLogger(ConnectionPool.class);
    private long MAX_IDLE_TIME = 300000L;
    private String driver;
    private String url;
    private String username;
    private String password;
    private String jdbcUrl;
    private int maxConnections;
    private boolean autoCommit = true;
    private boolean waitIfBusy;
    private Semaphore needConnection = new Semaphore(0);
    private boolean stop;
    private Stack<Connection> availableConnections;
    private Stack<Connection> busyConnections;
    private HashMap<Connection, Long> lastAccessTimeRecord = new HashMap();
    private String urlType = "";
    private DataSource datasource;
    private int transactionIsolation = 0;
    private Thread clenupThread;
    private Thread producerThread;

    public ConnectionPool(String driver, String url, String username, String password, int initialConnections, int maxConnections, boolean waitIfBusy) throws SQLException {
        this.driver = driver;
        this.url = url;
        this.username = username;
        this.password = password;
        this.urlType = "speratedURL";
        this.initialize(initialConnections, maxConnections, waitIfBusy);
    }

    public ConnectionPool(String driver, String jdbcUrl, int initialConnections, int maxConnections, boolean waitIfBusy, boolean autoCommit, int transactionIsolation) throws SQLException {
        this.driver = driver;
        this.jdbcUrl = jdbcUrl;
        this.urlType = "simpleURL";
        this.autoCommit = autoCommit;
        this.transactionIsolation = transactionIsolation;
        this.initialize(initialConnections, maxConnections, waitIfBusy);
    }

    public ConnectionPool(String driver, String jdbcUrl, int initialConnections, int maxConnections, boolean waitIfBusy) throws SQLException {
        this.driver = driver;
        this.jdbcUrl = jdbcUrl;
        this.urlType = "simpleURL";
        this.initialize(initialConnections, maxConnections, waitIfBusy);
    }

    public ConnectionPool(DataSource dataSource, int initialConnections, int maxConnections, boolean waitIfBusy) throws SQLException {
        this.urlType = "dataSource";
        this.datasource = dataSource;
        this.initialize(initialConnections, maxConnections, waitIfBusy);
    }

    public boolean isAutoCommit() {
        return this.autoCommit;
    }

    private void initialize(int initialConnections, int maxConnections, boolean waitIfBusy) throws SQLException {
        this.maxConnections = maxConnections;
        this.waitIfBusy = waitIfBusy;
        int sizeOfConnections = initialConnections > maxConnections ? maxConnections : initialConnections;
        this.availableConnections = new Stack();
        this.busyConnections = new Stack();
        for (int i = 0; i < sizeOfConnections; ++i) {
            Connection con = this.makeNewConnection();
            this.setTimeStamp(con);
            this.availableConnections.push(con);
        }
        this.producerThread = new Thread(new FillUpThread());
        this.producerThread.start();
        this.clenupThread = new Thread(new CleanUpThread());
        this.clenupThread.start();
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public synchronized Connection getConnection() throws SQLException {
        if (!this.availableConnections.isEmpty()) {
            Connection existingConnection = this.availableConnections.pop();
            if (existingConnection.isClosed()) {
                this.lastAccessTimeRecord.remove(existingConnection);
                this.notifyAll();
                return this.getConnection();
            }
            this.busyConnections.push(existingConnection);
            this.setTimeStamp(existingConnection);
            return existingConnection;
        }
        if (!this.waitIfBusy && this.busyConnections.size() >= this.maxConnections) {
            throw new SQLException("Connection limit reached");
        }
        if (this.busyConnections.size() < this.maxConnections) {
            this.needConnection.release();
        }
        try {
            this.wait();
            return this.getConnection();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        return this.getConnection();
    }

    private Connection makeNewConnection() throws SQLException {
        try {
            Class.forName(this.driver);
            Connection connection = this.urlType.equals("speratedURL") ? DriverManager.getConnection(this.url, this.username, this.password) : (this.urlType.equals("simpleURL") ? DriverManager.getConnection(this.jdbcUrl) : this.datasource.getConnection());
            connection.setTransactionIsolation(this.transactionIsolation);
            connection.setAutoCommit(this.autoCommit);
            return connection;
        }
        catch (ClassNotFoundException cnfe) {
            throw new SQLException("Can't find class for driver: " + this.driver);
        }
    }

    private synchronized void fillUpConnection(Connection conn) {
        this.setTimeStamp(conn);
        this.availableConnections.push(conn);
        this.notifyAll();
    }

    private void setTimeStamp(Connection connection) {
        this.lastAccessTimeRecord.put(connection, System.currentTimeMillis());
    }

    private boolean isConnectionStale(Connection connection) {
        long lastAccess;
        long currentTime = System.currentTimeMillis();
        return currentTime - (lastAccess = this.lastAccessTimeRecord.get(connection).longValue()) > this.MAX_IDLE_TIME;
    }

    private synchronized void closeStaleConnections() {
        Iterator iter = this.availableConnections.iterator();
        while (iter.hasNext()) {
            Connection existingConnection = (Connection)iter.next();
            if (!this.isConnectionStale(existingConnection)) continue;
            try {
                existingConnection.close();
                iter.remove();
            }
            catch (SQLException sql) {
                logger.error(sql.getMessage(), (Throwable)sql);
            }
        }
        iter = this.busyConnections.iterator();
        while (iter.hasNext()) {
            Connection busyConnection = (Connection)iter.next();
            if (!this.isConnectionStale(busyConnection)) continue;
            try {
                busyConnection.close();
                iter.remove();
                logger.warn("****Connection has checked out too long. Forced release. Check the program for calling release connection [free(Connection) method]");
            }
            catch (SQLException sql) {
                logger.error(sql.getMessage(), (Throwable)sql);
            }
        }
    }

    public synchronized void free(Connection connection) {
        this.busyConnections.removeElement(connection);
        this.availableConnections.addElement(connection);
        this.notifyAll();
    }

    public synchronized void dispose() {
        logger.info("Connection Pool Shutting down");
        this.stop = true;
        this.clenupThread.interrupt();
        this.producerThread.interrupt();
        this.closeConnections(this.availableConnections);
        this.availableConnections = new Stack();
        this.closeConnections(this.busyConnections);
        this.busyConnections = new Stack();
        this.lastAccessTimeRecord.clear();
        logger.info("All connection is closed");
        try {
            this.clenupThread.join();
            this.producerThread.join();
        }
        catch (Exception e) {
            logger.error("Cannot shutdown cleanup thread", (Throwable)e);
        }
        logger.info("Connection Pool Shutdown");
    }

    private void closeConnections(Stack<Connection> connections) {
        while (!connections.isEmpty()) {
            Connection connection = connections.pop();
            try {
                if (connection.isClosed()) continue;
                connection.close();
            }
            catch (SQLException sqle) {
                logger.warn(sqle.getMessage());
            }
        }
    }

    public synchronized String toString() {
        String info = "ConnectionPool(" + this.url + "," + this.username + "), available=" + this.availableConnections.size() + ", busy=" + this.busyConnections.size() + ", max=" + this.maxConnections;
        return info;
    }

    public void shutdown() throws SQLException {
        for (Connection c : this.availableConnections) {
            try {
                c.close();
            }
            catch (SQLException e) {
                logger.error("Error while closing the connection", (Throwable)e);
                throw new SQLException("Error while closing the connection", e);
            }
        }
        for (Connection c : this.busyConnections) {
            try {
                c.close();
            }
            catch (SQLException e) {
                logger.error("Error while closing the connection", (Throwable)e);
                throw new SQLException("Error while closing the connection", e);
            }
        }
    }

    class FillUpThread
    implements Runnable {
        FillUpThread() {
        }

        @Override
        public void run() {
            while (!ConnectionPool.this.stop) {
                try {
                    ConnectionPool.this.needConnection.acquire();
                    Connection conn = ConnectionPool.this.makeNewConnection();
                    ConnectionPool.this.fillUpConnection(conn);
                }
                catch (SQLException e) {
                    ConnectionPool.this.needConnection.release();
                    logger.error(e.getMessage(), (Throwable)e);
                }
                catch (InterruptedException e) {
                    logger.info("Fill up thread is interrupted to close");
                    break;
                }
            }
        }
    }

    class CleanUpThread
    implements Runnable {
        CleanUpThread() {
        }

        @Override
        public void run() {
            while (!ConnectionPool.this.stop) {
                try {
                    Thread.sleep(ConnectionPool.this.MAX_IDLE_TIME);
                    ConnectionPool.this.closeStaleConnections();
                }
                catch (InterruptedException e) {
                    logger.info("Clean up thread is interrupted to close");
                }
            }
        }
    }
}

