// Copyright 2006-2013 The Apache Software Foundation
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package org.apache.tapestry5.services;

import org.apache.tapestry5.ContentType;
import org.apache.tapestry5.MarkupWriter;
import org.apache.tapestry5.internal.structure.Page;

/**
 * Source for {@link org.apache.tapestry5.MarkupWriter} instances.
 */
public interface MarkupWriterFactory
{
    /**
     * Creates a markup writer for a particular content type.
     *
     * @param contentType type of content generated by the markup write; used to control the type of {@link
     *                    org.apache.tapestry5.dom.MarkupModel} used with the {@link org.apache.tapestry5.dom.Document}
     *                    the backs the markup writer.
     *                    
     * @deprecated use {@link #newMarkupWriter(Page)} instead which doesn't rely on the content type alone.
     */
    MarkupWriter newMarkupWriter(ContentType contentType);

    /**
     * Creates a markup writer for a particular content type, configured for <em>partial page rendering</em> (i.e., for
     * a response to an Ajax request).
     *
     * @param contentType type of content generated by the markup write; used to control the type of {@link
     *                    org.apache.tapestry5.dom.MarkupModel} used with the {@link org.apache.tapestry5.dom.Document}
     *                    the backs the markup writer.
     *
     * @deprecated use {@link #newPartialMarkupWriter(Page)} instead which doesn't rely on the content type alone.
     */
    MarkupWriter newPartialMarkupWriter(ContentType contentType);

    /**
     * Obtains a markup writer that will render the content for the provided logical page name.
     * Convenience method for {@link #newMarkupWriter(Page)}
     * 
     * @param pageName
     *            logical page name
     * @return writer configured for the page
     */
    MarkupWriter newMarkupWriter(String pageName);
    
    /**
     * Obtains a markup writer that will render the content for the provided logical page name,
     * configured for <em>partial page rendering</em> (i.e., for a response to an Ajax request).
     * Convenience method for {@link #newPartialMarkupWriter(Page)}
     * 
     * @param pageName
     *            logical page name
     * @return writer configured for the page
     * 
     * @since 5.4
     */
    MarkupWriter newPartialMarkupWriter(String pageName);
    
    /**
     * Obtains a markup writer that will render the content for the provided page. Takes into
     * account all necessary information such as the page's content type and doctype.
     * 
     * @param page the page to obtain a writer for
     * @return writer configured for the page
     * 
     * @since 5.4
     */
    MarkupWriter newMarkupWriter(Page page);
    
    /**
     * Obtains a markup writer that will render the content for the provided page,
     * configured for <em>partial page rendering</em> (i.e., for a response to an Ajax request).
     * Takes into account all necessary information such as the page's content type and doctype.
     * 
     * @param page
     *            the page to obtain a writer for
     * @return writer configured for the page
     * 
     * @since 5.4
     */
    MarkupWriter newPartialMarkupWriter(Page page);
}
