/*
 * ContentTemplate.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2002 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, suhler.
 *
 * Version:  2.2
 * Created by suhler on 99/06/28
 * Last modified by suhler on 02/11/14 14:27:54
 */

package sunlabs.brazil.template;

import java.util.Dictionary;

/**
 * Template class for extracting content out of remote html pages.
 * This class is used by the TemplateHandler, for extracting
 * the "content" out of html documents for later integration with
 * a look-and-feel template using one or more of:
 * {@link SetTemplate}, 
 * {@link BSLTemplate}, 
 * or
 * {@link sunlabs.brazil.filter.ReplaceFilter},
 *
 * The plan is to snag the title and the content, and put them into
 * request properties.  The resultant processed output will be
 * discarded.  The following properties are gathered:
 * <dl class=props>
 * <dt>title		<dd> The document title
 * <dt>all		<dd> The entire content
 * <dt>bodyArgs		<dd> The attributes to the body tag, if any
 * <dt>content		<dd> The body, delimited by content.../content>.
 *			The text inside multiple &lt;content&gt;
 *			... &lt;/content&gt; pairs
 *			are concatenated together.
 * <dt>script		<dd> All "&lt;script&gt;"..."&lt;/script&gt;"
 *			tags found in the document head
 * <dt>style		<dd> All "&lt;style"&gt;..."&lt;/style"&gt;
 *			tags found in the document head
 * <dt>meta-[name]	<dd> Every meta tag "name" and "content"
 * <dt>link-[rel]	<dd> Every link tag "rel" and "href"
 * <dt>user-agent	<dd> The origin user agent
 * <dt>referer		<dd> The user agent referrer (if any)
 * <dt>last-modified	<dd> The document last modified time (if any) in std format
 * <dt>content-length	<dd> The document content length, as fetched from the origin server
 * </dl>
 * Properties:
 * <dl class=props>
 * <dt>prepend	<dd>Prepend this string to the property names define above,
 * that are populated by this template. (defaults to "").
 * </dl>
 *
 * @author		Stephen Uhler
 * @version		%V% 2.2
 */

public class ContentTemplate extends Template {
    boolean inHead = true;
    String prefix;		// prefix all properties with this

    public boolean
    init(RewriteContext hr) {
	String all = hr.lex.rest();
	if (all != null) {
            prefix = hr.request.props.getProperty(hr.prefix + "prepend", "");
	    hr.request.props.put(prefix + "all", all);
	}
	return super.init(hr);
    }

    /**
     * Toss everything up to and including this entity.
     */

    public void
    tag_title(RewriteContext hr) {
	hr.reset();
    }

    /**
     * Gather up the title - no tags allowed between title .... /title.
     */

    public void
    tag_slash_title(RewriteContext hr) {
	hr.request.props.put(prefix + "title", hr.toString().trim());
	hr.reset();
    }

    /**
     * Append all "script" code while in the head section.
     */

    public void
    tag_script(RewriteContext hr) {
	if (inHead) {
	    boolean save = hr.accumulate(false);
	    hr.nextToken();
	    String script = hr.request.props.getProperty("script","") +
		    hr.getBody();
	    hr.request.props.put(prefix + "script",script);
	    hr.accumulate(save);
	    hr.reset();
	}
    }

    /**
     * Append all "style" code while in the head section.
     */

    public void
    tag_style(RewriteContext hr) {
	if (inHead) {
	    boolean save = hr.accumulate(false);
	    hr.nextToken();
	    String style = hr.request.props.getProperty("style","") +
		    hr.getBody();
	    hr.request.props.put(prefix + "style", style);
	    hr.accumulate(save);
	    hr.reset();
	}
    }

    /**
     * Mark end of head section.  All "script" content in the "body"
     * is left alone.
     */

    public void
    tag_slash_head(RewriteContext hr) {
	inHead = false;
    }

    /**
     * toss everything up to and including here, but turn on
     * content accumulation.
     */

    public void
    tag_content(RewriteContext hr) {
	hr.reset();
	hr.accumulate(true);
    }

    /**
     * Grab the "body" attributes, and toss all output to this point.
     */

    public void
    tag_body(RewriteContext hr) {
	inHead = false;
	String bodyArgs = hr.getArgs();
	if (bodyArgs != null) {
	    hr.request.props.put(prefix + "bodyArgs", bodyArgs);
	}
	hr.reset();
    }

    /**
     * Save the content gathered so far, and turn off content accumulation.
     */

    public void
    tag_slash_content(RewriteContext hr) {
	String content = hr.request.props.getProperty("content","") +
		hr.toString();
	hr.request.props.put(prefix + "content", content);
	hr.accumulate(false);
    }

    /**
     * If no content tags are present, use the entire "body" instead.
     */

    public void
    tag_slash_body(RewriteContext hr) {
	if (!hr.request.props.containsKey("content")) {
	    hr.request.props.put(prefix + "content", hr.toString());
	    hr.accumulate(false);
	}
    }

    /**
     * Extract data out of meta tags into the properties.
     * For "http-equiv" tags, set the corrosponding http respones header.
     */

    public void
    tag_meta(RewriteContext hr) {
	String name = hr.get("name", false);
	String equiv = hr.get("http-equiv", false);
	String content = hr.get("content", false);
	if ((name != null) && (content != null)) {
	    hr.request.props.put(prefix + "meta-" + name, content);
	} else if ((equiv != null) && (content != null)) {
	    hr.request.addHeader(equiv, content);
	}
    }

    /**
     * Extract data out of link tags into the properties.
     * Prefix the "rel" attribute with "link-" to use as the 
     * property name.
     */

    public void
    tag_link(RewriteContext hr) {
	String type = hr.get("rel", false);
	String href = hr.get("href", false);
	if ((type != null) && (href != null)) {
	    hr.request.props.put(prefix + "link-" + type, href);
	}
    }

    /**
     * Extract useful properties out of the http mime headers.
     */

    public boolean
    done(RewriteContext hr) {
	tag_slash_body(hr);

	transfer("user-agent", hr.request.headers, hr.request.props);
	transfer("referer", hr.request.headers, hr.request.props);
	transfer("last-modified", hr.request.responseHeaders, hr.request.props);
	transfer("content-length", hr.request.responseHeaders,hr.request.props);
	return true;
    }

    /**
     * Transfer an item to another hash table, if it exists
     */

    private boolean
    transfer (String key, Dictionary src, Dictionary dst) {
	Object obj = src.get(key);
	if (obj != null) {
	    dst.put(key, obj);
	    return true;
	} else {
	    return false;
	}
    }
}
