/*
 * Format.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 2000-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: cstevens.
 * Portions created by cstevens are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, drach, suhler.
 *
 * Version:  2.5
 * Created by cstevens on 00/04/17
 * Last modified by suhler on 04/11/30 14:59:57
 */

package sunlabs.brazil.util;

import java.util.Dictionary;
import java.util.Properties;

/**
 * Format a string by substituting values into it
 * from a properties object.
 *
 * @author colin stevens
 * @author stephen uhler
 */

public class Format {
    /**
     * Allow a property name to contain the value of another
     * property, permitting nested variable substitution in attribute
     * values.  The name of the embedded property to be substituted is
     * bracketted by "${" and "}".  See {@link #subst}.
     * <blockquote>
     * "ghi"		=> "foo"<br>
     * "deffoojkl"	=> "baz"<br>
     * "abcbazmno"	=> "garply"<br>
     * 
     * getProperty("ghi")			=> "foo"<br>
     * getProperty("def${ghi}jkl")		=> "baz"<br>
     * getProperty("abc${def${ghi}jkl}mno")	=> "garply"<br>
     * </blockquote>
     *
     * @param	props
     *		The table of variables to use when substituting.
     *
     * @param	expr
     *		The property name, prossibly containing substitutions.
     *
     * @param	defaultValue
     *		The value to use if the top-level substitution does not
     *		exist.  May be <code>null</code>.
     */

    public static String
    getProperty(Properties props, String expr, String defaultValue) {
	return props.getProperty(subst(props, expr), defaultValue);
    }

    /**
     * Allow a tag attribute value to contain the value of another
     * property, permitting nested variable substitution in attribute
     * values. To escape ${XXX}, use \${XXX}.
     * <p>
     * The sequence "\X" is identical to "X", except when "X" is one of:
     * <dl>
     * <dt>$<dd>A literal "$", that will not introduce a variable
     *      substitution if it is followed by "{".
     * <dt>n<dd>Insert a NL (newline) character
     * <dt>r<dd>Insert a CR (Carriage return) character
     * <dt>"<dd>Insert a single quote (").
     * <dt>l<dd>Insert a (<).
     * <dt>g<dd>Insert a (>).
     * <dt>a<dd>Insert a (&).
     * <dt>end of value<dd>Insert a "\".
     * </dl>
     * <p>
     * <blockquote>
     * "ghi"		= "foo"<br>
     * "deffoojkl"	= "baz"<br>
     * "abcbazmno"	= "garply"<br>
     * 
     * subst("ghi")			    = "ghi"<br>
     * subst("def${ghi}jkl")		    = "deffoojkl"<br>
     * subst("def\${ghi}jkl")		    = "def${ghi}jkl"<br>
     * subst("abc${def${ghi}jkl}mno")	    = "abcbazmno"<br>
     * subst("${abc${def${ghi}jkl}mno}")    = "garply"<br>
     * </blockquote>
     *
     * @param props	The table of variables to substitute.
     *			If this is a Properties object, then the
     *			getProperty() method is used instead of the
     *			Dictionary class get() method.
     * @param str	The expression containing the substitutions.
     *			Embedded property names, bracketted by "${" and "}" 
     *			are looked up in the props table and replaced with
     *			their value.  Nested substitutions are allowed. 
     *
     * @return		The substituted string.  If a variable is not 
     *			found in the table, the empty string is used.
     */

    public static String
    subst(Dictionary props, String str) {
	return subst(props, str, false);
    }

    /**
     * Allow a tag attribute value to contain the value of another
     * property, permitting nested variable substitution in attribute
     * values. To escape ${XXX}, use \${XXX}.
     * See {@link #subst(Dictionary props, String str) above}.
     * <p>
     * if <code>noEsc</code> is true, then
     * The sequence "\X" is identical to "\X" for all X except X=$.
     */

    public static String
    subst(Dictionary props, String str, boolean noEsc) {
	if (str == null) {
	    return null;
	}
	return subst(props, new Chars(str), noEsc);
    }


    private static String subst(Dictionary dict, Chars chars, boolean esc) {
	StringBuffer sb = new StringBuffer();
	char c;
	char save;
	String result;
	String value;

	loop: while (true) {
	    c = chars.get();
	    switch (c) {
	    case Chars.NUL:
		break loop;
	    case '$':
		c = chars.get();
		switch (c) {
		case Chars.NUL:
		    sb.append('$');
		    break loop;
		case '{':
		    save = chars.setend('}');
		    result = subst(dict, chars, esc);
		    chars.setend(save);
		    value = getProperty(dict, result);
		    sb.append(value);
		    break;
		default:
		    chars.pushback();
		    sb.append('$');
		    break;
		}
		break;
	    case '\\':
		c = chars.getraw();
		if (c == Chars.NUL) {
		    sb.append('\\');
		    break loop;
		}
		if (esc) {   // only \$ is special
		    switch (c) {
		    case '$':
			break;
		    default:
			chars.pushback();
			c = '\\';
			break;
		    }
		} else {	// other stuff is special too
		    switch (c) {
		    case 'a':
			c = '&';
			break;
		    case 'g':
			c = '>';
			break;
		    case 'l':
			c = '<';
			break;
		    case 'q':
			c = '"';
			break;
		    case 's':
			c = ' ';
			break;
		    case 'v':
			c = '\'';
			break;
		    case 'n':
			c = '\n';
			break;
		    case 'r':
			c = '\r';
			break;
		    case 't':
			c = '\t';
			break;
		    default:
			break;
		    }
		}
		sb.append(c);
		break;
	    default:
		sb.append(c);
		break;
	    }
	}
	return sb.toString();
    }

    private static String getProperty(Dictionary dict, String name) {
	int hash = name.indexOf('#');
	String def = "";
	
	if (hash >= 0) {
	    def = name.substring(hash + 1);
	    name = name.substring(0, hash);
	}

	Object obj;
	if (dict instanceof Properties) {
	    obj = ((Properties) dict).getProperty(name);
	} else {
	    obj = dict.get(name);
	}
	String value = (obj == null) ? null : obj.toString();

	if (value == null || value.length() == 0)
	    value = def;

	return value;
    }

    /**
     * See if a String represents a "true" boolean value, which consists of:
     * "yes", "true", "on", or "1", in any case.
     */

    public static boolean isTrue(String s) {
	if (s != null) {
	    String v = s.trim().toLowerCase();
	    return v.equals("true") || v.equals("yes") ||
	           v.equals("on") || v.equals("1");
	}
	return false;
    }

    /**
     * See if a String represents a "false" boolean value, which consists of:
     * "no", "false", "off", or "0", in any case.
     */

    public static boolean isFalse(String s) {
	if (s != null) {
	    String v = s.trim().toLowerCase();
	    return v.equals("false") || v.equals("no") ||
	           v.equals("off") || v.equals("0");
	}
	return false;
    }
}

class Chars {
    public static final char NUL = '\0';
    private int i;
    private char end;
    private char[] chars;

    Chars(String s) {
	i = 0;
	end = NUL;
	chars = s.toCharArray();
    }

    char getraw() {
	if (i >= chars.length)
	    return NUL;
	return chars[i++];
    }

    char get() {
	char c = getraw();
	if (c == end)
	    return NUL;
	return c;
    }

    void pushback() {
	if (--i < 0)
	    i = 0;
    }

    char setend(char end) {
	char save = this.end;
	this.end = end;
	return save;
    }
}
